;This program is used in the exercise in Chapter 13 to explore interrupts.
;The circuit for this exercise is simply a momentary switch connected to RA2
;with a weak pull-up resistor connected to the pin, the LCD connected to
;RA5 for display, and an LED connected to RC3.  The main program flashes the
;LED.  The push button will generate an external interrupt.  The interrupt
;service routine increments and displays the counter.  During the expercise
;the MPLAB IDE simulator function can be used to track the interrupt behavior.



	list      p=16F676       ; list directive to define processor
	#include <p16f676.inc>    ; processor specific variable definitions



	__CONFIG  _CP_OFF & _WDT_OFF & _BODEN & _PWRTE_ON & _INTRC_OSC_NOCLKOUT & _MCLRE_OFF & _CPD_OFF
;	__CONFIG  _CP_OFF & _WDT_OFF & _BODEN & _PWRTE_ON & _HS_OSC & _MCLRE_OFF & _CPD_OFF
; '__CONFIG' directive is used to embed configuration word within .asm file.
; The labels following the directive are located in the respective .inc file.
; See data sheet for additional information on configuration word settings.

;******************************************************************************
;Defines
;******************************************************************************

#define Bank0		0x00
#define	Bank1		0x80
#define	counter 	0x08

;*******************************************************************************
;LCD defines
;********************************************************************************
#define LCD			0x05			;pin RA5 connected to LCD data line
#define	LCD_BKSPC	0x08			;move cursor left
#define	LCD_RT		0x09			;move cursor right
#define	LCD_LF		0x0A			;move cursor down 1 line
#define LCD_CLS		0x0C			;clear LCD (allow 5 ms delay to complete)
#define	LCD_CR		0x0D			;move position 0 of next line
#define	LCD_BL_ON	0x11			;backlight on
#define	LCD_BL_OFF	0x12			;backlight off
#define LCD_OFF		0x15			;LCD off
#define LCD_ON1		0x16			;LCD on, cursor off, blink off
#define LCD_ON2		0x17			;LCD on, cursor off, blink on
#define LCD_ON3		0x18			;LCD on, cursor on, blink off
#define	LCD_ON4		0x19			;LCD on, cursor of, blink on
#define LCD_LINE0	0x80			;move to line 1, column 0, add position to 0x80
#define LCD_LINE1	0x94			;move to line 2, column 0, add position to 0x94
#define LCD_LINE2	0xA8			;move to line 3, column 0, add position to 0xA8
#define LCD_LINE3	0xBC			;move to line 4, column 0, add position to 0xBC
;end LCD defines
;******************************************************************************

;******************************************************************************
;General Purpose Registers (GPR's) 
;******************************************************************************

	cblock	0x20
	main_counter
	RA2_counter
	w_temp							;used in interrupt service routine to store
	status_temp						;pre-interrupt values for STATUS and w_reg
	count
	tempa
	bitcounter
	byte_to_send
	;********************************
	;used by Binary to BCD Converter
	countBCD
	temp
	h_byte
	l_byte
	r0
	r1
	r2
	;********************************
	endc


;******************************************************************************
;Reset Vector 
;******************************************************************************
	ORG     0x000         	; processor reset vector
	nop						; required by in circuit debugger  
	goto    Init            ; go to beginning of program

;******************************************************************************
;Interrupt Vector     
;******************************************************************************
	ORG	0x004
	goto	 interrupt_service
	return			; interrupt trap - returns without re-enabling
	
;******************************************************************************
;Initialization
;******************************************************************************
Init
	BANKSEL	Bank1
	call    0x3FF      ; retrieve factory calibration value
	movwf	OSCCAL

	BANKSEL	Bank0		;select bank0
	clrf	PORTA		;clear port bus
	clrf	PORTC
	movlw	b'00000111'	;comparator disconnected
	movwf	CMCON		;
	movlw	b'00010000'	;globals off,peripherals off, RA2 external interrupt enabled
						;interrupt flags cleared
	movwf	INTCON
	
	BANKSEL	Bank1		; BANK1
	clrf	OPTION_REG	;enable weak pull-ups

	movlw	b'00000100'	;weak pull-up on RA2
	movwf	WPUA		;
	movlw	b'00000100'	;RA2 set as input, others output 
	movwf	TRISA		;program PORTA
	movlw	b'00000000'	;all output
	movwf	TRISC		;program PORTC
	movlw	b'00000000'	;all digital I/O
	movwf	ANSEL		
	BANKSEL	Bank0		;back to bank0
;end pic initialization
;*****************************************************************************
	
;*****************************************************************************
;Initialize the LCD display
;*****************************************************************************
	bsf		PORTA,LCD			;set to high for resting state
	call	wait1sec		;to allow LCD to inialize
	movlw	LCD_CLS
	call	LCDOutput
	call	delay5mS		;to allow LCD to complete clear
;end LCD initialization
;****************************************************************************


display_text
	movlw	.7					;there are 7 characters in "RA2 = "
	movwf	countBCD			;countBCD is used as a counter for using the table
	clrf	temp				;temp is used as a offset location within table
display_loop
	call	text_display
	call	LCDOutput			;this call puts the character on the LCD
	incf	temp				;increment the offset to the next charater location
	decfsz	countBCD			;decrement the table counter, continue if not zero
	goto	display_loop

	clrf	h_byte				;clear these variables
	clrf	l_byte
	bsf		INTCON,GIE			;enable global interrupts
	
main
	movlw	b'00001000'			;puts a "1" in the bit position of RC3 in PORTC
	xorwf	PORTC				;if RC# is high, then low; if low, the high
	call	wait250mS			;delay to flash LED at 250mS interval
	goto	main

;**************************************************************************
;This is the interrupt service routine that is called when an interrupt
;is generated.  If there are multiple interrupt sources possible, then
;early in this routine you would need to do a bit check of the verious 
;interrupt flags to determing which source generated the interrupt and then
;take appropriate action.  In this program, there is only one interrupt
;enabled (RA2/INT) so there is no need to do this check, it is assumed that
;a high to low transistion on RA2 is the source of the interrupt.
;**************************************************************************

interrupt_service
	bcf		INTCON,INTE			;disable addtional RA2 interrupts
	movwf	w_temp				;copy contents of w_reg into a temp register
	swapf	STATUS,w			;swap the nibbles of STATUS and place into
								;the w_register, these nibbles will be swapped
								;back when the STATUS register is recovered
								;at the end of the interrupt service routine
	BANKSEL	Bank0				;forces a return to Bank 0 regardless of bank 
								;when interrupt occurred
	movwf	status_temp			;put the swapped old STATUS reg value in a temp
								;location, now with w and STATUS registers praotected
								;proceed with the interrupt service

 	movlw	LCD_LINE0+.6		;move cursor to display the RA2 count
	call	LCDOutput
	incf	RA2_counter
	movfw	RA2_counter
	movwf	l_byte
	call	display_DEC

	swapf	status_temp,w		;swap the nibbles in status_temp and put result in w_reg
	movwf	STATUS				;STATUS now returned to pre-interrupt value
	swapf	w_temp,f			;take the old value of w_reg and swap nibbles
	swapf	w_temp,w			;swap nibbles again and place into w_reg, w_reg
								;now returned to pre-interrupt value
	bcf		INTCON,INTF			;clear RA2 interrupt flag before enabling the interrupt
	bsf		INTCON,INTE			;enable RA2 interrupts

	retfie						;this command also sets GIE to enable global interrupts
;end interrupt service
;*******************************************************************************

;*****************************************************************************
; 
;use this routine to call the static text that is contained in table.
;the programming technique avoids a problem with using the program counter
;to jump across memory pages.  The PCL is only 8 bits so any offset that goes
;beyond 255 will cause problems because of a jump across a memory page.  This 
;code checks if the PCL+offset goes over 255 by checking the carry bit.  If the
;carry bit is set, 1 is added to the high byte of the program counter PCLATH to jump
;the next page.
;
;Put the offset value in temp and then call text_display.
;
;*****************************************************************************
text_display
	movlw	LOW table			;get low byte of table's location
	addwf	temp,w				;add the offset to the low byte
	movlw	HIGH table			;get the high byte of table's location
	btfsc	STATUS,C			;check to see if a page has been crossed
	addlw	1					;if so, add 1 to the high byte
	movwf	PCLATH				;put the adjusted high byte into counter
	movfw	temp				;put offset in the w register
	call	table_get			;jump into the table
	return						;from text_display

;*****************************************************************************
;
;Display 10-bit decimal number
;put high byte in h_byte
;put low byte in l_byte
;call display_DEC
;
;*****************************************************************************
display_DEC
	call	b2_BCD
	swapf	r1,w
	andlw	b'00001111'
	addlw	.48
	call	LCDOutput
	movfw	r1
	andlw	b'00001111'
	addlw	.48
	call	LCDOutput
	swapf	r2,w
	andlw	b'00001111'
	addlw	.48
	call	LCDOutput
	movfw	r2
	andlw	b'00001111'
	addlw	.48
	call	LCDOutput
	return
;end display_DEC
;*******************************************************************************



;*****************************************************************************
;
;LCD Output
;put the desired value into the w register and call LCDOutput
;
;*****************************************************************************
LCDOutput
	movwf	byte_to_send
	movlw .8
	movwf bitcounter	;set up to send 8 bits
	bcf		PORTA,LCD			;send start bit
	call	bitdelay
nextbit
	rrf		byte_to_send,f	;sending lsb first
	btfss	STATUS,C
	bcf		PORTA,LCD
	btfsc	STATUS,C
	bsf		PORTA,LCD
	call	bitdelay
	decfsz	bitcounter,f
	goto	nextbit
	bsf		PORTA,LCD		;set to high for resting state
	call	delay5mS
	return
;end LCDOutput
;*******************************************************************************


;******************************************************************************
;
;Binary to BCD converter
;Put binary value to be converted into h_byte and l_byte
;then call b2_BCD.  Decimal converstion is returned in r0, r1, and r2
;with MSB in lower nibble of r0.
;
;******************************************************************************
b2_BCD
	bcf		STATUS,C
	movlw	.16
	movwf	countBCD
	clrf	r0
	clrf	r1
	clrf	r2
loop16
	rlf		l_byte,f
	rlf		h_byte,f
	rlf		r2,f
	rlf		r1,f
	rlf		r0,f
	
	decfsz	countBCD,f
	goto	adjDEC
	retlw	0

adjDEC
	movlw	r2
	movwf	FSR
	call	adjBCD
	
	movlw	r1
	movwf	FSR
	call	adjBCD

	movlw	r0
	movwf	FSR
	call	adjBCD

	goto	loop16

adjBCD
	movlw	3
	addwf	0,w
	movwf	temp
	btfsc	temp,3
	movwf	0
	movlw	30
	addwf	0,w
	movwf	temp
	btfsc	temp,7
	movwf	0
	retlw	0
;end Binary to BCD Converter
;******************************************************************************	



;******************************************************************************
;Delay Routines
;******************************************************************************
;more delay options than needed, this is standard delay code I use in other programs
wait5sec
		call	wait1sec
		call	wait1sec
		call	wait1sec
		call	wait1sec		;1 seconds plus the one to follow
		

wait1sec
		call	wait300mS
		call	wait300mS
		call	wait300mS
		goto	wait100mS
wait300mS
        call    delay50mS
wait250mS
        call    delay50mS
        call    delay50mS
        call    delay50mS
wait100mS
        call    delay50mS
wait50mS
        call    delay50mS
		return
;delay1ms, is a very accurate 1mS delay for a 4Mhz clock.
delay1mS
        movlw   .198
        movwf   count
        nop
        goto    $+1
        goto    $+1
dly1mS        
        goto    $+1
        decfsz  count, F
        goto    dly1mS
        return
;delay5mS uses delay1mS to get a very accurate 5 mS delay
delay5mS
        call    delay1mS
        call    delay1mS
        call    delay1mS
        call    delay1mS
        call    delay1mS
        movlw   .4
        movwf   count
tweek5mS
        decfsz  count, F
        goto    tweek5mS
        return
;delay50mS uses delay1mS to get a very accurate 50mS delay
delay50mS
        movlw   .50
        movwf   tempa
dly50mS
        call    delay1mS
        decfsz  tempa, F
        goto    dly50mS
        movlw   .14
        movwf   count
tweek50mS
        decfsz  count, F
        goto    tweek50mS
        return
;delay200mS uses delay1mS to get a very accurate 200mS delay.
delay200mS
        movlw   .200
        movwf   tempa
dly200mS
        call    delay1mS
        decfsz  tempa, F
        goto    dly200mS
        movlw   .64
        movwf   count
tweek200mS
        decfsz  count, F
        goto    tweek200mS
        return

bitdelay
		movlw	.30				;this number works if the user uses the calbrated
								;value for the internal clock. This routine, including
								;the goto and nop statements below allow the user to develop
								;an anticipated delay of 100uS for the bits at 9600 Baud.
								;This delay can be verified by using the stop watch function
								;of MPLab Simulator
		movwf	count
		goto    $+1
       
bit
		decfsz	count, f
		goto	bit
		return
;end delay rountines


;******************************************************************************
;
;data
;
;*****************************************************************************

table_get
	addwf	PCL,f		;add the offset to the program counter to jump to character
table
	dt	LCD_LINE0,'R','A','2',' ', '=' ,' '
	




	end
